import store from 'systems/store.js'

let draggedIndex = null


function inject () {
  store.onNotice('pathFinder', ['waypoints changed', 'path changed'], refreshWaypointsCards)
}



function refreshWaypointsCards (pathFinder) {
  clearWaypointsCards()

  pathFinder.waypoints.forEach((zone, index) => {
    addWaypointCard(zone, index)

    if (pathFinder.pathes[index]) {
      addDistanceCard(pathFinder.pathes[index].distance)
    }
  })

  if (pathFinder.pathes.length > 0) {
    let totalDistance = pathFinder.pathes.reduce((sum, p) => sum + p.distance, 0)
    addTotalDistanceCard(totalDistance)
  }
}


function addWaypointCard (zone, index) {
  let container = document.querySelector('#path-panel .waypoints-section')

  let card = document.createElement('div')
  let zoneName = zone.infos[0] ? zone.infos[0].content : zone.id

  card.className = 'waypoint-card'
  card.draggable = true
  card.dataset.index = index

  card.addEventListener('dragstart', () => draggedIndex = index)
  card.addEventListener('dragover', e => e.preventDefault())
  card.addEventListener('drop', () => reorderWaypoints(index))

  card.innerHTML = `
    <span class="drag-handle">⋮⋮</span>
    <span class="waypoint-id">${zoneName}</span>
    <button class="remove">✕</button>
  `

  card.querySelector('.remove').onclick = () => removeWaypoint(zone.id)

  container.appendChild(card)
}


function addDistanceCard (distance) {
  let container = document.querySelector('#path-panel .waypoints-section')
  let card = document.createElement('div')
  card.className = 'distance-card'
  card.innerHTML = `<span>${Math.round(distance)}m</span>`
  container.appendChild(card)
}


function addTotalDistanceCard (totalDistance) {
  let container = document.querySelector('#path-panel .waypoints-section')
  let card = document.createElement('div')
  card.className = 'total-distance-card'
  card.innerHTML = `<span>Total: ${Math.round(totalDistance)}m</span>`
  container.appendChild(card)
}


function reorderWaypoints (targetIndex) {
  if (draggedIndex === null || draggedIndex === targetIndex) return

  store.doWith(['pathFinder'], ({ pathFinder }) => {
    let [moved] = pathFinder.waypoints.splice(draggedIndex, 1)
    pathFinder.waypoints.splice(targetIndex, 0, moved)

    store.mutate('pathFinder', pf => pf.waypoints = pathFinder.waypoints)
    store.notifyChange('pathFinder', 'waypoints changed')
  })

  draggedIndex = null
}


function removeWaypoint (zoneId) {
  store.doWith(['pathFinder'], ({ pathFinder }) => {
    const index = pathFinder.waypoints.findIndex(z => z.id === zoneId)
    if (index !== -1) {
      store.mutate('pathFinder', pf => pf.waypoints.splice(index, 1))
      store.notifyChange('pathFinder', 'waypoints changed')
    }
  })
}


function clearWaypointsCards () {
  document.querySelector('#path-panel .waypoints-section').innerHTML = ''
}





export default {
  id: 'pathWaypointsUi',
  dependencies: ['pathPanel'],
  tags: ['pathFinder', 'UI'],
  sendTo: {
    pathPanel: {
      className: 'waypoints-section',
      order: 50,
      inject
    }
  },
  ressources: {
    css: ['path_finder/ui/path_waypoints']
  }
}