import fs    from 'systems/fs.js'
import store from 'systems/store.js'



async function loadQuarter (uri, pos) {
  let blob = await new Promise((resolve, reject) => {
    window.SafExplorer.getImageTileBytes(uri, pos, resolve, reject)
  })

  let imgUrl = URL.createObjectURL(blob)

  let img = new Image()
  img.src = imgUrl

  await new Promise((resolve) => { img.onload = resolve })

  return img
}


async function startLinearLoad (uri, i = 0) {
  let x = i % 4
  let y = Math.floor(i / 4)

  let img = await loadQuarter(uri, [x, y])

  store.mutate('mapAdder', mapAdder => mapAdder.quarterImages[x][y] = img)
  store.notifyChange('mapAdder', 'quarter image loaded', x, y)

  if (i < 4*4 - 1) {
    startLinearLoad(uri, i + 1)
  } else {
    store.mutate('mapAdder', mapAdder => mapAdder.quartersLoaded = true)
    store.notifyChange('mapAdder', 'quarter images loaded')
  }
}


function setQuartersImages () {
  return Array.from({ length: 4 }, () =>
    Array.from({ length: 4 }, () => null)
  )
}


function loadQuarters (itemName) {
  store.mutate('mapAdder', mapAdder => {
    mapAdder.quarterImages  = setQuartersImages()
    mapAdder.quartersLoaded = false
    mapAdder.mapName        = itemName.split('.')[0]
  })


  store.notifyChange('mapAdder', 'quarter images initiated')

  let uri = fs.getUri('add_map/' + itemName)

  startLinearLoad(uri)
}


function getAddMapImages (callback) {
  let path = 'add_map'

  fs.directory(path, {
    onSuccess: callback,
    onError: (error) => {
      throw new Error('Failed load directory add_map/ :' + error)
    }
  })
}


function readmeLoad () {
  alert(`
QUARTER LOAD HELP

0) Purpose:
   - Allows loading very large images that may not work with the normal loader
   - WARNING: "Load image" may still work, but with reduced quality and the extracted result may look wrong or broken

1) Image location:
   - Place your image inside the 'add_map' folder
   - Clicking "Quarter Load" will automatically create this folder if it doesn't exist
   - The folder must be located at the root of your resources
     (next to projects/, maps/, etc.)

2) How to use:
   - Use the "Quarter Load" button instead of "Load image"
   - You can follow the guide provided by the "Help" button
   - Note: don’t forget to move/copy your plan into the add_map folder first
`)
}


function handleFolderCreation () {
  fs.directory('add_map', {
    method: 'POST',
    onSuccess: () => {
      alert('add_map/ folder created successfully. Please place your images in the folder and try again.')
    },
    onError: () => {
      alert('Failed to create add_map/ folder. Please create it manually and place your images inside.')
    }
  })
}

function handleImageSelection (items) {
  if (items.length === 1) {
    loadQuarters(items[0].name)
    return
  }

  let message = 'Choose image (type number):\n' + items.map((item, i) =>
    `${i + 1}. ${item.name}`
  ).join('\n')

  let choice = prompt(message, '1')
  let index = parseInt(choice) - 1

  if (index >= 0 && index < items.length) {
    loadQuarters(items[index].name)
  }
}

function quarterLoad () {
  getAddMapImages(items => {
    if (!items) {
      handleFolderCreation()
      return
    }

    if (items.length === 0) {
      alert('The add_map/ folder is empty. Please place your images inside and try again.')
      return
    }

    handleImageSelection(items)
  })
}


function createSection (section) {
  let title = document.createElement('h3')
  title.textContent = 'Mobile - large images'
  section.appendChild(title)

  let buttonContainer = document.createElement('div')
  buttonContainer.className = 'button-container'

  let buttons = [
    { text: 'Load',   onClick: quarterLoad },
    { text: 'Readme', onClick: readmeLoad }
  ]

  buttons.forEach(({ text, onClick }) => {
    let button = document.createElement('button')
    button.textContent = text
    button.className = 'text'
    button.onclick = onClick
    buttonContainer.appendChild(button)
  })

  section.appendChild(buttonContainer)
}


export default {
  id: 'addMapQuarterLoad',
  tags: ['addMap', 'data'],
  dependencies: ['addMapMain'],
  sendTo: window.cordova ? {
    addMapPanel: {
      type: 'section',
      className: 'quarter-section',
      onCreate: createSection
    },
    ressourcesSnapshot: {
      storeNoticeCleaners: [
        { storeName: 'mapsTiles', notice: 'map tiles created' },
        { storeName: 'maps',      notice: 'map created' }
      ]
    }
  } : null
}