
import fullScreenAction  from 'systems/full_screen_action.js'
import store             from 'systems/store.js'
import ressourcesHandler from 'utilities/helpers/ressources_handler.js'
import fs                from 'systems/fs.js'


// TODO - check crash_graph_fixer / doublon of full screen behavior -> update to use only one system



function extract ({ mapAdder }) {
  if (!mapAdder.image && !mapAdder.quartersLoaded) {
    alert('No image loaded. Please add an image first.')
    return
  }


  initFullScreenFeedback()
  writeMetaFile((folderName, meta) => {
    generateAndWriteTiles(folderName, meta, () => {
      store.notifyChange('mapsTiles', 'map tiles created', folderName)
      store.notifyChange('maps',      'map created',       folderName)
      fullScreenAction.setTitle('Done. You can refresh.')
      fullScreenAction.setShowRefresh(true)
    })
  })
}








function writeMetaFile (callback) {
  ensureFreeFolder(({ error, folderName, mapId }) => {
    if (error) {
      return triggerError(error)
    }

    let meta = generateMetaJson(mapId)
    ressourcesHandler.addMapMeta(folderName, meta, () => {
      fullScreenAction.setTitle('Meta written, start generating tiles.')
      callback(folderName, meta)
    })
  })
}




function ensureFreeFolder (callback) {
  store.doWith(['mapAdder', 'maps', 'projectData'], ({ mapAdder, maps, projectData }) => {
    let mapId = toId(mapAdder.mapName)
    let folderName = mapId + '_' + toId(projectData.meta['grid-system'])

    if (!isIdFree(mapId, maps, projectData)) {
      return callback({ error: 'A map with same id and grid system already exists.' })
    } else if (!isFolderNotUsedByAnotherMap(folderName, maps)) {
      return callback({ error : 'The auto-suggested folder name is already taken by another map.'})
    }

    checkFolderIsFree(folderName, {
      doIfFree:    () => callback({ folderName, mapId }),
      doIfNotFree: () => callback({ error: 'You already have a folder "' + folderName + '" in your maps folder. It is not a proper Jafamap "map folder" and conflicts with the auto-suggested folder name.' })
    })
  })
}


function isIdFree (mapId, maps, projectData) {
  let gridSystem = projectData.meta['grid-system']

  for (let id in maps) {
    let isSameId         = (id === mapId)
    let isSameGridSystem = (maps[id].gridSystem === gridSystem)

    if (isSameGridSystem && isSameId) {
      return false
    }
  }

  return true
}


function isFolderNotUsedByAnotherMap (folderName, maps) {
  for (let id in maps) {
    if (maps[id].folderName === folderName) {
      return false
    }
  }

  return true
}


function checkFolderIsFree (folderName, { doIfFree, doIfNotFree }) {
  fs.directory('maps/' + folderName, {
    onSuccess: (data) => {
      if (data === null || 404) {
        doIfFree()
      } else {
        doIfNotFree()
      }
    },
    onError: (error) => {
      throw new Error('Failed during checking maps/ folder : ' + error)
    }
  })
}


function getSize (mapAdder) { // TODO - quick & dirty duplication from ./controls
  if (mapAdder.image) {
    return mapAdder.image
  }

  let first = mapAdder.quarterImages[0][0]

  return {
    width:  first.width  * 4,
    height: first.height * 4
  }
}



function generateMetaJson (mapId) {
  return store.doWith(['mapAdder', 'projectData'], ({ mapAdder, projectData }) => {
    let projectMeta = projectData.meta
    let imgSize = getSize(mapAdder)
    let meta = {
      name: mapAdder.mapName,
      id: mapId,
      range: {
        start: {
          x: Math.floor(mapAdder.x / projectMeta.size.width),
          y: Math.floor(mapAdder.y / projectMeta.size.height)
        },
        end: {
          x: Math.floor(((mapAdder.x - 0.001) + imgSize.width  * mapAdder.z) / projectMeta.size.width),
          y: Math.floor(((mapAdder.y - 0.001) + imgSize.height * mapAdder.z) / projectMeta.size.height)
        },
        ignore: []
      }
    }

    meta['grid-system'] = projectMeta['grid-system']

    return meta
  })
}








function generateAndWriteTiles (folderName, meta, callback) {
  store.doWith(['mapAdder', 'projectData'], ({ mapAdder, projectData }) => {
    let tilePxSize = getTilePxSize({ mapAdder, projectData })
    let canvas     = new OffscreenCanvas(tilePxSize.width, tilePxSize.height)
    let ctx        = canvas.getContext('2d')
    let tileInfo   = toBuildTileInfo({ mapAdder, projectData, meta })

    loopGenerateAndWrite({ tileInfo, folderName, ctx }, callback)
  })
}


function toBuildTileInfo ({ mapAdder, projectData, meta }) {
  let start      = meta.range.start
  let size       = projectData.meta.size
  let imageScale = mapAdder.z
  let totalX     = (meta.range.end.x + 1 - start.x)
  let totalY     = (meta.range.end.y + 1 - start.y)
  let totalTile  = totalX * totalY

  let offset = {
    x: Math.round((mapAdder.x - meta.range.start.x * size.width)  / imageScale),
    y: Math.round((mapAdder.y - meta.range.start.y * size.height) / imageScale)
  }

  return {
    totalX, totalTile, offset, size, imageScale, start,
    image: mapAdder.image,
    quarterImages: mapAdder.quarterImages
  }
}


function loopGenerateAndWrite ({ tileInfo, folderName, ctx }, onDone, i = 0) {
  let { totalX, totalTile, offset, start, image, quarterImages } = tileInfo

  let ix = i % totalX
  let iy = Math.floor(i / totalX)

  let wx = start.x + ix
  let wy = start.y + iy
  let fileName = wx + '-' + wy + '.jpg'


  updateCanvas(ctx, { image, quarterImages }, { ix, iy, offset })

  postCanvas(ctx.canvas, folderName, fileName, () => {
    i++
    if (i == totalTile) {
      onDone()
    } else {
      fullScreenAction.setTitle('Generating tiles ' + i + '/' + totalTile)
      loopGenerateAndWrite({ tileInfo, folderName, ctx }, onDone, i)
    }
  })
}





function singleImageUpdateCanvas (ctx, image, { ix, iy, offset }) {
  let { width, height } = ctx.canvas
  ctx.clearRect(0, 0, width, height)

  ctx.drawImage(
    image,
    ix * width  - offset.x,
    iy * height - offset.y,
    width,
    height,
    0, 0, width, height
  )
}




function ijToXy (i, j) { // TODO - duplication with ./controls
  return {
    x: i % 2 * 2 + j % 2,
    y: Math.floor(i / 2) * 2 + Math.floor(j / 2)
  }
}


function forEachQuarter (quarterImages, action) {
  for (let i = 0; i < 4; i++) {
    for (let j = 0; j < 4; j++) {
      let img = quarterImages[i][j]

      action(img, i, j)
    }
  }
}

function forEachCollidingQuarter (quarterImages, { ix, iy, offset, width, height }, action) {
  forEachQuarter(quarterImages, (img, i, j) => {
    let quarterPos = ijToXy(i, j)

    let sx = ix * width  - offset.x - quarterPos.x * img.width
    let sy = iy * height - offset.y - quarterPos.y * img.height
    let isOut = (sx + width <= 0 || sy + height <= 0 || sx >= img.width || sy >= img.height)

    if (!isOut) {
      action(img, sx, sy)
    }
  })
}

function quartersImageUpateCanvas (ctx, quarterImages, { ix, iy, offset }) {
  let { width, height } = ctx.canvas
  ctx.clearRect(0, 0, width, height)

  forEachCollidingQuarter(quarterImages, { ix, iy, offset, width, height }, (img, sx, sy) => {
    let clipL = Math.max(0, -sx)
    let clipT = Math.max(0, -sy)
    let clipR = Math.max(0, sx + width - img.width)
    let clipB = Math.max(0, sy + height - img.height)

    ctx.drawImage(
      img,
      sx + clipL, sy + clipT, width - clipL - clipR, height - clipT - clipB,
      clipL, clipT, width - clipL - clipR, height - clipT - clipB
    )
  })
}


function updateCanvas (ctx, { image, quarterImages }, targetTileParam) {
  if (image) {
    singleImageUpdateCanvas(ctx, image, targetTileParam)
  } else {
    quartersImageUpateCanvas(ctx, quarterImages, targetTileParam)
  }
}


function postCanvas (canvas, folderName, fileName, callback) {
  canvasToJpgForm(fileName, canvas).then(formData => {
    fs.file('maps/' + folderName + '/' + fileName, {
      method: 'POST',
      type: 'blob',
      body: formData,
      onSuccess: callback,
      onError: (error) => console.error(error)
    })
  })
}


function canvasToJpgForm (name, canvas) {
  return canvas.convertToBlob({ type: 'image/jpeg', quality: 1 }).then(blob => {
    let formData = new FormData()
    formData.append('file', blob, name)

    return formData
  })
}


function getTilePxSize ({ mapAdder, projectData }) {
  return {
    width:  Math.round(projectData.meta.size.width  / mapAdder.z),
    height: Math.round(projectData.meta.size.height / mapAdder.z)
  }
}












function initFullScreenFeedback () {
  fullScreenAction.show({
    withIcon: true,
    isALoading: true,
    title: 'Map adder : check & init'
  })
}


function triggerError (text) {
  fullScreenAction.setIsALoading(false)
  fullScreenAction.setHasAnError(true)
  fullScreenAction.setTitle(text)
  fullScreenAction.setShowRefresh(true)
}


function toId (string) {
  return string
    .trim()
    .toLowerCase()
    .replace(/\s+/g, '_')
    .replace(/[^a-z0-9_]/g, '')
}




let extractButton = {
  text: 'extract',
  onClick: store.makeDependentTo(['mapAdder'], extract)
}

export default {
  id: 'addMapExtract',
  tags: ['addMap', 'data'],
  dependencies: ['addMapMain', 'mapsMetaData', 'projectData'],
  sendTo: {
    addMapPanel: extractButton,
    ressourcesSnapshot: {
      storeNoticeCleaners: [
        { storeName: 'mapsTiles', notice: 'map tiles created' },
        { storeName: 'maps',      notice: 'map created' }
      ]
    }
  }
}